/**
 * AAA-Modellarttool
 *
 * The class in this file implements the ShapeChange Target interface to 
 * generate and load the 3AM files.
 *
 * (c) 2009-2012 Arbeitsgemeinschaft der Vermessungsverwaltungen der 
 * Länder der Bundesrepublik Deutschland (AdV)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact:
 * interactive instruments GmbH
 * Trierer Strasse 70-72
 * 53115 Bonn
 * Germany
 */

package de.adv_online.aaa.modellarttool;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Set;

import de.interactive_instruments.ShapeChange.Options;
import de.interactive_instruments.ShapeChange.ShapeChangeResult;
import de.interactive_instruments.ShapeChange.Model.ClassInfo;
import de.interactive_instruments.ShapeChange.Model.Info;
import de.interactive_instruments.ShapeChange.Model.Model;
import de.interactive_instruments.ShapeChange.Model.PackageInfo;
import de.interactive_instruments.ShapeChange.Model.PropertyInfo;
import de.interactive_instruments.ShapeChange.Model.EA.ClassInfoEA;
import de.interactive_instruments.ShapeChange.Model.EA.PackageInfoEA;
import de.interactive_instruments.ShapeChange.Model.EA.PropertyInfoEA;

public class ModellartRep {

	private PackageInfo pi = null;
	private Model model = null;
	private Options options = null;
	private ShapeChangeResult result = null;
	private String modellart = "";
	private static String AAAZielVersion = "6.0.1";
	private boolean set = false;

	private HashSet<ClassInfo> processed = new HashSet<ClassInfo>();
	private HashSet<ClassInfo> written = new HashSet<ClassInfo>();
	private HashSet<Info> Members = new HashSet<Info>();

	public ModellartRep(PackageInfo p, Model m, Options o,
			ShapeChangeResult r, String ma) {
		pi = p;
		model = m;
		options = o;
		result = r;
		modellart = ma;

		for (ClassInfo ci : model.classes(pi)) {
			LoadClass(ci);
		}
		
		set = true;
	}
	
	private void LoadClass(ClassInfo ci) {
		int cat = ci.category();
		switch(cat) {
		case Options.UNKNOWN:
		case Options.FEATURE:
		case Options.UNION:
		case Options.DATATYPE:
		case Options.MIXIN:
		case Options.OBJECT:
			if (!processed.contains(ci)) {
				processed.add(ci);
				if (inModel(ci)) {
					Members.add(ci);
					for (String st : ci.supertypes()) {
						ClassInfo cis = model.classById(st);
						if (cis!=null && cis.inSchema(pi) && !processed.contains(cis)) {
							LoadClass(cis);
						}
					}
					for (PropertyInfo propi : ci.properties().values()) {
						if (inModel(propi)) {
							ClassRequired(propi.typeInfo().id);
							Members.add(propi);
							// Bei einer Relation auch die inverse Rolle
							PropertyInfo rpropi = propi.reverseProperty();
							if (rpropi!=null) {
								Members.add(rpropi);
							}
						}
					}
				}
			}
			break;
		case Options.ENUMERATION:
		case Options.CODELIST:
			if (!processed.contains(ci)) {
				processed.add(ci);
				if (inModel(ci)) {
					Members.add(ci);
					for (PropertyInfo propi : ci.properties().values()) {
						if (inModel(propi)) {
							Members.add(propi);
						}
					}
				}
			}
			break;
		default:
			result.addWarning("Klasse mit unbekannter Art wurde nicht verarbeitet: "+ci.name());
			break;
		}
	}

	private boolean inModel(Info i) {
		String s1 = i.taggedValue("AAA:Modellart");
		if (s1==null || s1.trim().length()==0)
			return true;
		String[] sa = s1.split(",");
		for (String s2 : sa) {
			if (s2.equals(name()))
				return true;
		}
		return false;
	}
	
	private void addToModel(Info i) {
		String s1 = i.taggedValue("AAA:Modellart");
		if (s1==null || s1.trim().length()==0) {
			/*
			if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.PackageInfoEA")) {
				PackageInfoEA iea = (PackageInfoEA) i;
				iea.taggedValue("AAA:Modellart",name());
			} else if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.ClassInfoEA")) {
				ClassInfoEA iea = (ClassInfoEA) i;
				iea.taggedValue("AAA:Modellart",name());
			} else if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.PropertyInfoEA")) {
				PropertyInfoEA iea = (PropertyInfoEA) i;
				iea.taggedValue("AAA:Modellart",name());
			} else {
				result.addInfo("Unsupported Java class: "+i.getClass().getName());
			}
			*/
			return;
		}
		String[] sa = s1.split(",");
		for (String s2 : sa) {
			if (s2.equals(name()))
				return;
		}
		if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.PackageInfoEA")) {
			PackageInfoEA iea = (PackageInfoEA) i;
			iea.taggedValue("AAA:Modellart",s1+","+name());
		} else if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.ClassInfoEA")) {
			ClassInfoEA iea = (ClassInfoEA) i;
			iea.taggedValue("AAA:Modellart",s1+","+name());
		} else if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.PropertyInfoEA")) {
			PropertyInfoEA iea = (PropertyInfoEA) i;
			iea.taggedValue("AAA:Modellart",s1+","+name());
		} else {
			result.addInfo("Unsupported Java class: "+i.getClass().getName());
		}
	}
	
	private void removeFromModel(Info i) {
		String s1 = i.taggedValue("AAA:Modellart");
		if (s1==null || s1.trim().length()==0)
			return;
		String[] sa = s1.split(",");
		s1 = "";
		for (String s2 : sa) {
			if (!s2.equals(name())) {
				if (s1.length()==0)
					s1 = s2;
				else
					s1 += "," + s2;
			}
		}
		if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.PackageInfoEA")) {
			PackageInfoEA iea = (PackageInfoEA) i;
			iea.taggedValue("AAA:Modellart",s1);
		} else if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.ClassInfoEA")) {
			ClassInfoEA iea = (ClassInfoEA) i;
			iea.taggedValue("AAA:Modellart",s1);
		} else if (i.getClass().getName().equals("de.interactive_instruments.ShapeChange.Model.EA.PropertyInfoEA")) {
			PropertyInfoEA iea = (PropertyInfoEA) i;
			iea.taggedValue("AAA:Modellart",s1);
		} else {
			result.addInfo("Unsupported Java class: "+i.getClass().getName());
		}
	}
	
	public boolean contains(Info i) {
		if (Members.contains(i))
			return true;
		return false;
	}
	
	private void SetClass(ClassInfo ci) {
		int cat = ci.category();
		switch(cat) {
		case Options.UNKNOWN:
		case Options.FEATURE:
		case Options.UNION:
		case Options.DATATYPE:
		case Options.MIXIN:
		case Options.OBJECT:
			if (!processed.contains(ci)) {
				processed.add(ci);
				Members.add(ci);
				for (String st : ci.supertypes()) {
					ClassInfo cis = model.classById(st);
					if (cis!=null && cis.inSchema(pi) && !processed.contains(cis)) {
						SetClass(cis);
					}
				}
				for (PropertyInfo propi : ci.properties().values()) {
					SetProperty(propi, false, false);
				}
			}
			break;
		case Options.ENUMERATION:
		case Options.CODELIST:
			if (!processed.contains(ci)) {
				processed.add(ci);
				Members.add(ci);
				for (PropertyInfo propi : ci.properties().values()) {
					SetProperty(propi, false, true);
				}
			}
			break;
		default:
			result.addWarning("Klasse mit unbekannter Art wurde nicht verarbeitet: "+ci.name());
			break;
		}
	}

	private void ClassRequired(String classId) {
		ClassInfo ci = model.classById(classId);
		if (ci!=null && ci.inSchema(pi)) {
			int cat = ci.category();
			switch(cat) {
			case Options.UNKNOWN:
			case Options.UNION:
			case Options.DATATYPE:
			case Options.MIXIN:
			case Options.FEATURE:
			case Options.OBJECT:
			case Options.ENUMERATION:
			case Options.CODELIST:
				SetClass(ci);
				break;
			default:
				result.addWarning("Klasse mit unbekannter Art wurde nicht verarbeitet: "+ci.name());
				break;
			}
		}
	}
	
	private void SetProperty(PropertyInfo propi, boolean force, boolean codedValue) {
		boolean export = (force || inModel(propi));
		if (!codedValue)
			export = export || propi.cardinality().minOccurs>0;
		export = export && propi.name().length()>0;
		if (export) {
			Members.add(propi);
			if (!codedValue) {
				ClassRequired(propi.typeInfo().id);
				if (!propi.isAttribute() && propi.isNavigable()) {
					// Bei einer Relation sind auch alle inversen Rollen automatisch Teil der Modellart
					PropertyInfo rpropi = propi.reverseProperty();
					if (rpropi!=null) {
						Members.add(rpropi);
					}
				}
			} else if (!Members.contains(propi.inClass()))
				Members.add(propi.inClass());
		}
	}	
	
	public void writeToFile(String filename) {
		if (!set) {
			result.addError("Export der Datei nicht möglich, da die Einträge noch nicht definiert sind.");
			return;
		}
		
		try {
			PrintWriter out = null;
			written.clear();
			try {
				out = new PrintWriter(new FileWriter(filename));
				out.println("AFIS-ALKIS-ATKIS-Modellartinhalt");
				out.println("Version: "+AAAZielVersion);
				out.println("Name: "+modellart);
				out.println("#");
				
				Set<ClassInfo> classes = model.classes(pi);
				ClassInfo[] classesArr = new ClassInfo[classes.size()];
				classes.toArray(classesArr);					
				Arrays.sort(classesArr, new Comparator<ClassInfo>() {
					public int compare(ClassInfo ci1, ClassInfo ci2) {
						return ci1.name().compareTo(ci2.name());
					}});
				for ( int cidx = 0; cidx < classesArr.length; cidx++ ) {
						ClassInfo k = classesArr[cidx];
						DumpClass(out, k);
					}
			} finally {
				if (out!=null)
					out.close();
				result.addInfo("Schreiben der Datei abgeschlossen.");
			}
		} catch (IOException e){
			result.addError("Problem beim Schreiben von Datei: "+filename);
			e.printStackTrace();
		}		
	}
	
	private void AddEntry(PrintWriter out, ClassInfo i, boolean included) {
		String s = i.name()+" --- Objektart/Datentyp";
		if (!included)
			s = "# "+s;
		out.println(s);
		result.addDebug(s);
	}
	
	private void AddEntry(PrintWriter out, PropertyInfo i, boolean included, boolean codedValue) {
		String s = i.inClass().name()+"/";
		if (codedValue) {
			String iv = i.initialValue();
			if (iv!=null && iv.trim().length()>0)
				s += iv.trim();
			else
				s += i.name();
		} else {
			s += i.name();
		}
		int cat = i.inClass().category();
		if (cat==Options.ENUMERATION || cat==Options.CODELIST)
			s += " --- Werteart";
		else if (i.isAttribute())
			s += " --- Attributart";
		else
			s += " --- Relationsart";
		if (!included)
			s = "# "+s;
		out.println(s);
		result.addDebug(s);
	}
	
	private void DumpProperty(PrintWriter out, PropertyInfo propi, boolean codedValue) {
		if (propi.name().length()==0)
			return;
		if (Members.contains(propi)) {
			AddEntry(out, propi, true, codedValue);
		} else {
			AddEntry(out, propi, false, codedValue);
		}
	}

	private void DumpClass(PrintWriter out, ClassInfo ci) {
		int cat = ci.category();
		switch(cat) {
		case Options.MIXIN:
		case Options.UNKNOWN:
		case Options.FEATURE:
		case Options.UNION:
		case Options.DATATYPE:
		case Options.OBJECT:
			if (!written.contains(ci)) {
				if (Members.contains(ci)) {
					AddEntry(out, ci, true);
				} else {
					AddEntry(out, ci, false);
				}
				written.add(ci);
				for (PropertyInfo propi : ci.properties().values()) {
					DumpProperty(out, propi, false);
				}
			}
			break;
		case Options.ENUMERATION:
		case Options.CODELIST:
			if (!written.contains(ci)) {
				written.add(ci);
				for (PropertyInfo propi : ci.properties().values()) {
					DumpProperty(out, propi, true);
				}
				break;
			}
		default:
			result.addWarning("Klasse mit unbekannter Art wurde nicht verarbeitet: "+ci.name());
			break;
		}
	}
	
	public void writeToModel() {
		if (!set) {
			result.addError("Übernahme in das Modell nicht möglich, da die Einträge noch nicht definiert worden sind.");
			return;
		}
		
		written.clear();
		Set<ClassInfo> classes = model.classes(pi);
		for (ClassInfo k : classes) {
			WriteClassToModel(k);
		}
	}
	
	private void WritePropertyToModel(PropertyInfo propi) {
		if (propi.name().length()==0)
			return;
		if (Members.contains(propi)) {
			addToModel(propi);
		} else {
			removeFromModel(propi);
		}
		// Bei einer Relation sind auch alle inversen Rollen automatisch Teil der Modellart
		PropertyInfo rpropi = propi.reverseProperty();
		if (rpropi!=null) {
			if (Members.contains(rpropi)) {
				addToModel(rpropi);
			} else {
				removeFromModel(rpropi);
			}
		}
	}

	private void WriteClassToModel(ClassInfo ci) {
		int cat = ci.category();
		switch(cat) {
		case Options.MIXIN:
		case Options.UNKNOWN:
		case Options.FEATURE:
		case Options.UNION:
		case Options.DATATYPE:
		case Options.OBJECT:
			if (!written.contains(ci)) {
				if (Members.contains(ci)) {
					addToModel(ci);
				} else {
					removeFromModel(ci);
				}
				written.add(ci);
				for (PropertyInfo propi : ci.properties().values()) {
					WritePropertyToModel(propi);
				}
			}
			break;
		case Options.ENUMERATION:
		case Options.CODELIST:
			if (!written.contains(ci)) {
				if (Members.contains(ci)) {
					addToModel(ci);
				} else {
					removeFromModel(ci);
				}
				written.add(ci);
				for (PropertyInfo propi : ci.properties().values()) {
					WritePropertyToModel(propi);
				}
				break;
			}
		default:
			result.addWarning("Klasse mit unbekannter Art wurde nicht verarbeitet: "+ci.name());
			break;
		}
	}
	
	public void clearInModel() {
		Set<ClassInfo> classes = model.classes(pi);
		for (ClassInfo ci : classes) {
			removeFromModel(ci);
			for (PropertyInfo propi : ci.properties().values()) {
				removeFromModel(propi);
				// Bei einer Relation auch die inverse Rolle
				PropertyInfo rpropi = propi.reverseProperty();
				if (rpropi!=null) {
					removeFromModel(rpropi);
				}
			}
		}
	}
	
	public String name() {
		return modellart;
	}

	public ModellartRep(PackageInfo p, Model m, Options o,
			ShapeChangeResult r, FileReader reader) {
		pi = p;
		model = m;
		options = o;
		result = r;

		try {
			BufferedReader input =  new BufferedReader(reader);
			try {
				String line = null;
				
				line = input.readLine();
				if (line==null || !line.trim().contains("AFIS-ALKIS-ATKIS-Modellartinhalt")) {
					result.addError("Die Datei beginnt nicht mit dem erwateten Wert 'AFIS-ALKIS-ATKIS-Modellartinhalt'. Der Ladevorgang wurde abgebrochen.");
					input.close();
					return;
				}

				line = input.readLine();
				if (line==null || !line.trim().startsWith("Version: ")) {
					result.addError("Es wurde keine Version angegeben. Der Ladevorgang wurde abgebrochen.");
					input.close();
					return;
				} else {
					String s = line.trim().substring(9);
					if (!s.equals(AAAZielVersion)) {
						result.addError("Die Version "+s+" passt nicht zur erwarteten Version "+AAAZielVersion+". Der Ladevorgang wurde abgebrochen.");
						input.close();
						return;						
					}
				}
				
				line = input.readLine();
				if (line==null || !line.trim().startsWith("Name: ")) {
					result.addError("Es wurde kein Modellartname angegeben. Der Ladevorgang wurde abgebrochen.");
					input.close();
					return;
				} else {
					String s = line.trim().substring(6);
					modellart = s.trim();
				}
				
				for (ClassInfo ci : model.classes(pi)) {
					LoadClass(ci);
				}

				processed.clear();
				
				while ((line = input.readLine()) != null) {
					line = line.trim();
					if (!line.startsWith("#")) {
						String[] s = line.split(" --- ");
						if (s.length!=2) {
							result.addError("Zeile mit unerlaubter Syntax: '"+line+"'");
						} else {
							s[0] = s[0].trim();
							s[1] = s[1].trim();
							if (s[1].equals("Objektart/Datentyp")) {
								ClassInfo cix = model.classByName(s[0]); // FIXME multiple classes with the same name; do it for each...
								if (cix!=null && cix.inSchema(pi)) {
									SetClass(cix);
								} else {
									result.addError("Objektart/Datentyp "+s[0]+" kann der Modellart nicht zugeordnet werden, die Klasse wurde nicht im Anwendungsschema gefunden.");
								}
							} else if (s[1].equals("Attributart") || s[1].equals("Relationsart") || s[1].equals("Werteart")) {
								String[] s2 = s[0].split("/",2);
								if (s2.length!=2) {
									result.addError("Zeile mit unerlaubter Syntax: '"+line+"'");
								} else {
									s2[0] = s2[0].trim();
									s2[1] = s2[1].trim();
									ClassInfo cix = model.classByName(s2[0]); // FIXME multiple classes with the same name; do it with each...
									if (cix!=null && cix.inSchema(pi)) {
										SetClass(cix);
										boolean found = false;
										for (PropertyInfo propi : cix.properties().values()) {
											String sx1 = propi.name();
											String sx2 = propi.initialValue();
											if (sx1!=null && sx1.toLowerCase().equals(s2[1].toLowerCase())) {
												SetProperty(propi, true, false);
												found = true;
												break;
											} else if (s[1].equals("Werteart") && sx2!=null && sx2.toLowerCase().equals(s2[1].toLowerCase())) {
												SetProperty(propi, true, true);
												found = true;
												break;
											}
										}
										if (!found) {
											result.addError(s[1]+" "+s2[1]+" zu Objektart/Datentyp "+s2[0]+" kann der Modellart nicht zugeordnet werden, die Eigenschaft wurde nicht im Anwendungsschema gefunden.");
										}
									} else {
										result.addError("Objektart/Datentyp "+s2[0]+" kann der Modellart nicht zugeordnet werden, die Klasse wurde nicht im Anwendungsschema gefunden.");
									}
								}
							}
						}						
					}
				}
			} finally {
				input.close(); 
				result.addInfo("Laden der Datei abgeschlossen.");
				set = true;
			}
		}
		catch (IOException e){
			result.addError("Problem beim Lesen von Modellartdatei.");
			e.printStackTrace();
		}
	}	
}
